# 3. Develop Your Application Service

After you have entered the relevant information as per the aforementioned process, before proceeding with testing and submitting for review, you need to develop your application service. It interacts with Shoplazza through a series of APIs.

## Obtain Authorization for Shoplazza API

### Overview

Public applications use OAuth 2.0 for Authorization, and the interaction process is shown in the figure below.

> You need to pay attention to Step 2, 3, 4, 7, 8, 9 and then you need to write your processing logic code in Step 3, 4, 9.

> ⚠️ **Warning:** authorization_code can only be used once.

## Important Steps Detail Info

### Step 2: The Shoplazza backend establishes communication and authorization through the APP URL

After Merchant adds app from the [APP Store](https://appstore.shoplazza.com/), the Shoplazza services will send a request by APP URL which is configured by the [partner center](https://partners.shoplazza.com/).

For example:
```
https://stamped-review.apps.shoplazza.com/api/auth?hmac=c4caf9b08bdeff7531bb12712ffea860264ec24f5fd953832505c5024d19edca&install_from=app_store&shop=rwerwre.myshoplaza.com&store_id=1339409
```

The format is:
```
app_url?hmac=${hmac}&install_from=app_store&shop=${store_domain_name}&store_id=${store_id}
```

### Step 3: Security Checks by APP Service

Before we continue, make sure your app performs the following security checks. If any of the checks fail, then your app must reject the request with an error, and must not continue.

- The hmac is valid and signed by Shoplazza.
- The shop parameter is a valid shop hostname, ends with myshoplaza.com.

For Security Checks, [SHOPLAZZA OAuth SDKs](https://www.shoplazza.dev/reference/api-libraries) also has corresponding methods. You can quickly verify hmac and shop parameter by SHOPLAZZA OAuth SDKs.

**Example using OAuth-SDK-Go:**
```go
import (
  co "github.com/shoplazza-os/oauth-sdk-go"
  "github.com/shoplazza-os/oauth-sdk-go/shoplazza"
)

oauth := &co.Config{
                ClientID:     "s1Ip1WxpoEAHtPPzGiP2rK2Az-P07Nie7V97hRKigl4",
                ClientSecret: "0LFJcNqVb2Z1nVt9xT72vOo0sTWd6j8wVX60Y5xdzZZ",
                Endpoint:     shoplazza.Endpoint,
                RedirectURI:  "https://3830-43-230-206-233.ngrok.io/oauth_sdk/redirect_uri/",
                Scopes:       []string{"read_shop"},
        }
oauth.ValidShop("xxx.myshoplaza.com")        // verify shop parameter

var requestUrl = "http://example.com/some/redirect_uri?code={authorization_code}&shop={store_name}.myshoplaza.com&hmac={hmac}"
query := strings.Split(requestUrl, "?")
params, _ := url.ParseQuery(query[1])
oauth.SignatureValid(params)                 // verify hmac
```

#### HMac Validation Detail

Every request or redirect from Shoplazza to your app's server includes an hmac parameter that can be used to verify the authenticity of Shoplazza. For each request, you must remove the hmac entry from the query string and process it through an **HMAC-SHA256** hash function.

For example, for the following request:
```
http://example.com/some/redirect/uri?code=1vtke5ljOOL2jPds6gM0TNCeYZDitYB&shop=simon.myshoplaza.com&hmac=22bad22eee1f92836f7773e87d973479
```

To remove the hmac, you can transform the query string to a map, remove the hmac key-value pair, and then lexicographically concatenate your map back to a query string. This leaves the remaining parameters from the example query string:
```
code=1vtke5ljOOL2jPds6gM0TNCeYZDitYB&shop=simon.myshoplaza.com
```

##### Process the hash function

After you remove hmac and reformat the query string, you can process the string through an HMAC-SHA256 hash function using the **Client secret** as the secret key.

### Step 4: Pass the Scopes to obtain permission to access the APIs you need

The application service receives this request for authorization. If there are no issues with the authorization, it will redirect using the 302 method.

For example:
```
https://rwerwre.myshoplaza.com/admin/oauth/authorize?client_id=ZwwR8eXIOq0Rr2XN3zUywxc0q-S9C3w4VkH3HnNUL_Q&redirect_uri=https%3A%2F%2Fstamped-review.apps.shoplazza.com%2Fapi%2Fauth%2Fcallback&response_type=code&scope=read_product+read_order+read_collection+read_shop+read_script_tags+write_script_tags+read_customer+read_price_rules+read_comments
```

Format:
```
https://${store_domain_name}/admin/oauth/authorize?client_id={client_id}&scope={scopes}&redirect_uri={redirect_uri}&response_type={response_type}&state={state}
```

Parameters:
- `store_domain_name`: `${store_name}.myshoplaza.com`
- `store_name`: The name of merchant's store
- `client_id`: The Client ID for the app. You can get it at https://partners.shoplazza.com/ after registering the app
- `scope`: A space separated list of [scopes]
- `redirect_uri`: The URL to which a merchant is redirected after authorizing the app
- `response_type`: The response type of OAuth 2.0 process, here we need to fill in `code`
- `state`: A random value, used to prevent CSRF attacks

#### Scopes

Scope represents the permissions that a public application needs to request, with different permissions allowing access to different APIs. **The value of the scope is chosen and assembled by the developer** into the 302 redirect URL mentioned above. On the merchant's app installation confirmation page, the merchant will be asked to authorize the required permissions.

### Step 7: Get Access_token by Code

You can get an Access token by the following Request:
```
POST https://{store_name}.myshoplaza.com/admin/oauth/token
```

In this request, `store_name` is the name of the merchant's store along with the following parameters:

- `client_id`: The Client ID for the App
- `client_secret`: The Client secret key for the App
- `code`: The authorization_code provided in the redirect
- `grant_type`: The grant type of OAuth 2.0 process, please fill in `authorization_code` here
- `redirect_uri`: The redirect_uri of the App

The server responds with an access token:

```json
{
  "token_type": "Bearer",
  "expires_at": 1550546245,
  "access_token": "eyJ0eXAiOiJKV1QiLCJh",
  "refresh_token": "def502003d28ba08a964e",
  "store_id": "2",
  "store_name": "xiong1889"
}
```

Response parameters:
- `token_type`: It will just return Bearer
- `expires_at`: The access_token expired time, in timestamp
- `access_token`: The correct access_token
- `refresh_token`: The refresh token used to refresh the access_token if needed
- `store_id`: Store's ID in Shoplazza
- `store_name`: Store name

> **OAuth SDK Available!**<br>
Similarly, you can quickly get an access token by SHOPLAZZA OAuth SDKs.

**Example using OAuth-SDK-Go:**
```go
token, err := oauth.Exchange("xxx")
if err != nil {
    // handle error
}
fmt.Println(token.AccessToken)
```

## Making API Calls

Once you have obtained the access token, you can use it to make API calls to Shoplazza. Include the access token in the Authorization header of your API requests:

```
Authorization: Bearer {access_token}
```

Use the access token to access the APIs that correspond to the scopes you requested during the authorization process.